/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.jackrabbit.oak.blob.cloud.azure.blobstorage;

import com.azure.identity.ClientSecretCredential;
import com.azure.storage.blob.BlobContainerClient;
import com.azure.storage.blob.sas.BlobSasPermission;
import com.azure.storage.blob.specialized.BlockBlobClient;
import com.azure.storage.common.policy.RequestRetryOptions;
import com.microsoft.aad.msal4j.MsalServiceException;

import org.apache.commons.lang3.reflect.MethodUtils;
import org.apache.jackrabbit.core.data.DataStoreException;
import org.junit.After;
import org.junit.Before;
import org.junit.ClassRule;
import org.junit.Test;

import java.lang.reflect.Field;
import java.net.URISyntaxException;
import java.security.InvalidKeyException;
import java.time.OffsetDateTime;
import java.util.Properties;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

public class AzureBlobContainerProviderTest {

    @ClassRule
    public static AzuriteDockerRule azurite = new AzuriteDockerRule();

    private static final String CONTAINER_NAME = "test-container";
    private AzureBlobContainerProvider provider;

    @Test
    public void testBuilderWithConnectionString() {
        String connectionString = getConnectionString();
        
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();
        
        assertNotNull("Provider should not be null", provider);
        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
        assertEquals("Connection string should match", connectionString, provider.getAzureConnectionString());
    }

    @Test
    public void testBuilderWithAccountNameAndKey() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withAccountKey("testkey")
                .build();
        
        assertNotNull("Provider should not be null", provider);
        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
    }

    @Test
    public void testBuilderWithServicePrincipal() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();
        
        assertNotNull("Provider should not be null", provider);
        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
    }

    @Test
    public void testBuilderWithSasToken() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withSasToken("sas-token")
                .withBlobEndpoint("https://testaccount.blob.core.windows.net")
                .build();
        
        assertNotNull("Provider should not be null", provider);
        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
    }

    @Test
    public void testBuilderInitializeWithProperties() {
        Properties properties = new Properties();
        properties.setProperty(AzureConstants.AZURE_CONNECTION_STRING, getConnectionString());
        properties.setProperty(AzureConstants.AZURE_STORAGE_ACCOUNT_NAME, "testaccount");
        properties.setProperty(AzureConstants.AZURE_STORAGE_ACCOUNT_KEY, "testkey");
        properties.setProperty(AzureConstants.AZURE_TENANT_ID, "tenant-id");
        properties.setProperty(AzureConstants.AZURE_CLIENT_ID, "client-id");
        properties.setProperty(AzureConstants.AZURE_CLIENT_SECRET, "client-secret");
        properties.setProperty(AzureConstants.AZURE_SAS, "sas-token");
        properties.setProperty(AzureConstants.AZURE_BLOB_ENDPOINT, "https://testaccount.blob.core.windows.net");
        
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .initializeWithProperties(properties)
                .build();
        
        assertNotNull("Provider should not be null", provider);
        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
        assertEquals("Connection string should match", getConnectionString(), provider.getAzureConnectionString());
    }

    @Test
    public void testGetBlobContainerWithConnectionString() throws DataStoreException {
        String connectionString = getConnectionString();
        
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();
        
        BlobContainerClient containerClient = provider.getBlobContainer();
        assertNotNull("Container client should not be null", containerClient);
        assertEquals("Container name should match", CONTAINER_NAME, containerClient.getBlobContainerName());
    }

    @Test
    public void testGetBlobContainerWithRetryOptions() throws DataStoreException {
        String connectionString = getConnectionString();
        RequestRetryOptions retryOptions = new RequestRetryOptions();
        Properties properties = new Properties();
        
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();
        
        BlobContainerClient containerClient = provider.getBlobContainer(retryOptions, properties);
        assertNotNull("Container client should not be null", containerClient);
        assertEquals("Container name should match", CONTAINER_NAME, containerClient.getBlobContainerName());
    }

    @Test
    public void testBuilderWithNullContainerName() {
        // Builder accepts null container name - no validation
        AzureBlobContainerProvider.Builder builder = AzureBlobContainerProvider.Builder.builder(null);
        assertNotNull("Builder should not be null", builder);
    }

    @Test
    public void testBuilderWithEmptyContainerName() {
        // Builder accepts empty container name - no validation
        AzureBlobContainerProvider.Builder builder = AzureBlobContainerProvider.Builder.builder("");
        assertNotNull("Builder should not be null", builder);
    }

    @Test
    public void testGenerateSharedAccessSignatureWithConnectionString() {
        String connectionString = getConnectionString();

        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();

        try {
            String sas = provider.generateSharedAccessSignature(
                    null,
                    "test-blob",
                    new BlobSasPermission().setReadPermission(true),
                    3600,
                    new Properties()
            );
            assertNotNull("SAS token should not be null", sas);
            assertFalse("SAS token should not be empty", sas.isEmpty());
        } catch (Exception e) {
            // Expected for Azurite as it may not support all SAS features
            assertTrue("Should be DataStoreException, URISyntaxException, or InvalidKeyException",
                    e instanceof DataStoreException ||
                    e instanceof URISyntaxException ||
                    e instanceof InvalidKeyException);
        }
    }

    @Test
    public void testGetBlobContainerWithInvalidConnectionString() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString("invalid-connection-string")
                .build();

        try {
            provider.getBlobContainer();
            fail("Expected exception with invalid connection string");
        } catch (Exception e) {
            // Expected - can be DataStoreException or IllegalArgumentException
            assertNotNull("Exception should not be null", e);
            assertTrue("Should be DataStoreException or IllegalArgumentException",
                    e instanceof DataStoreException || e instanceof IllegalArgumentException);
        }
    }

    @Test
    public void testGetBlobContainerWithServicePrincipalMissingCredentials() throws DataStoreException{
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                // Missing client secret
                .build();

            BlobContainerClient containerClient = provider.getBlobContainer();
            // May succeed with incomplete credentials - Azure SDK might handle it differently
            assertNotNull("Container client should not be null", containerClient);
    }

    @Test(expected = DataStoreException.class)
    public void testGetBlobContainerWithSasTokenMissingEndpoint() throws DataStoreException{
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withSasToken("sas-token")
                // Missing blob endpoint
                .build();

            BlobContainerClient containerClient = provider.getBlobContainer();
    }

    @Test
    public void testBuilderChaining() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString("connection1")
                .withAccountName("account1")
                .withAccountKey("key1")
                .withBlobEndpoint("endpoint1")
                .withSasToken("sas1")
                .withTenantId("tenant1")
                .withClientId("client1")
                .withClientSecret("secret1")
                .build();

        assertNotNull("Provider should not be null", provider);
        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
        assertEquals("Connection string should match", "connection1", provider.getAzureConnectionString());
    }

    @Test
    public void testBuilderWithEmptyStrings() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString("")
                .withAccountName("")
                .withAccountKey("")
                .withBlobEndpoint("")
                .withSasToken("")
                .withTenantId("")
                .withClientId("")
                .withClientSecret("")
                .build();

        assertNotNull("Provider should not be null", provider);
        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
        assertEquals("Connection string should be empty", "", provider.getAzureConnectionString());
    }

    @Test
    public void testInitializeWithPropertiesEmptyValues() {
        Properties properties = new Properties();
        properties.setProperty(AzureConstants.AZURE_CONNECTION_STRING, "");
        properties.setProperty(AzureConstants.AZURE_STORAGE_ACCOUNT_NAME, "");
        properties.setProperty(AzureConstants.AZURE_STORAGE_ACCOUNT_KEY, "");

        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .initializeWithProperties(properties)
                .build();

        assertNotNull("Provider should not be null", provider);
        assertEquals("Connection string should be empty", "", provider.getAzureConnectionString());
    }

    @Test
    public void testInitializeWithPropertiesNullValues() {
        Properties properties = new Properties();
        // Properties with null values (getProperty returns default empty string)

        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .initializeWithProperties(properties)
                .build();

        assertNotNull("Provider should not be null", provider);
        assertEquals("Connection string should be empty", "", provider.getAzureConnectionString());
    }

    @Test
    public void testGetBlobContainerServicePrincipalPath() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        try {
            BlobContainerClient containerClient = provider.getBlobContainer();
            assertNotNull("Container client should not be null", containerClient);
        } catch (Exception e) {
            // Expected for invalid service principal credentials
            assertTrue("Should be DataStoreException or related",
                    e instanceof DataStoreException || e.getCause() instanceof Exception);
        }
    }

    @Test
    public void testGetBlobContainerSasTokenPath() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withSasToken("sv=2020-08-04&ss=b&srt=sco&sp=rwdlacx&se=2023-12-31T23:59:59Z&st=2023-01-01T00:00:00Z&spr=https&sig=test")
                .withBlobEndpoint("https://testaccount.blob.core.windows.net")
                .build();

        try {
            BlobContainerClient containerClient = provider.getBlobContainer();
            assertNotNull("Container client should not be null", containerClient);
        } catch (Exception e) {
            // Expected for invalid SAS token
            assertTrue("Should be DataStoreException or related",
                    e instanceof DataStoreException || e.getCause() instanceof Exception);
        }
    }

    @Test
    public void testGenerateSharedAccessSignatureServicePrincipalPath() throws DataStoreException, URISyntaxException, InvalidKeyException{
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        try {
            String sas = provider.generateSharedAccessSignature(
                    null,
                    "test-blob",
                    new BlobSasPermission().setReadPermission(true),
                    3600,
                    new Properties()
            );

            //fail if no exception is thrown
            fail("Expected exception for invalid service principal credentials");
        } catch (MsalServiceException e) {
            // Expected for invalid service principal credentials
            //Message should be: AADSTS900023: Specified tenant identifier 'tenant-id' is neither a valid DNS name, nor a valid external domain
          assertTrue("Message should be different", e.getMessage().contains("AADSTS900023: Specified tenant identifier 'tenant-id' is neither a valid DNS name, nor a valid external domain"));
            // Accept any exception as authentication will fail with invalid credentials
        }
    }

    @Test
    public void testGenerateSharedAccessSignatureAccountKeyPath() throws DataStoreException, URISyntaxException, InvalidKeyException {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withAccountKey("testkey")
                .build();

            String sas = provider.generateSharedAccessSignature(
                    null,
                    "test-blob",
                    new BlobSasPermission().setReadPermission(true),
                    3600,
                    new Properties()
            );
            assertNotNull("SAS token should not be null", sas);
            assertFalse("SAS token should not be empty", sas.isEmpty());
    }

    @Test
    public void testBuilderStaticMethod() {
        AzureBlobContainerProvider.Builder builder = AzureBlobContainerProvider.Builder.builder("test-container");
        assertNotNull("Builder should not be null", builder);

        AzureBlobContainerProvider provider = builder.build();
        assertNotNull("Provider should not be null", provider);
        assertEquals("Container name should match", "test-container", provider.getContainerName());
    }

    @Test
    public void testBuilderConstructorAccess() throws Exception {
        // Test that Builder constructor is private by accessing it via reflection
        java.lang.reflect.Constructor<AzureBlobContainerProvider.Builder> constructor =
                AzureBlobContainerProvider.Builder.class.getDeclaredConstructor(String.class);
        assertFalse("Constructor should not be public", constructor.isAccessible());

        // Make it accessible and test
        constructor.setAccessible(true);
        AzureBlobContainerProvider.Builder builder = constructor.newInstance("test-container");
        assertNotNull("Builder should not be null", builder);
    }

    @Test
    public void testDefaultEndpointSuffixUsage() throws Exception {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        // Use reflection to access the DEFAULT_ENDPOINT_SUFFIX constant
        Field defaultEndpointField = AzureBlobContainerProvider.class.getDeclaredField("DEFAULT_ENDPOINT_SUFFIX");
        defaultEndpointField.setAccessible(true);
        String defaultEndpoint = (String) defaultEndpointField.get(null);
        assertEquals("Default endpoint should be core.windows.net", "core.windows.net", defaultEndpoint);

        // Test that the endpoint is used in service principal authentication
        RequestRetryOptions retryOptions = new RequestRetryOptions();

        try {
            MethodUtils.invokeMethod(provider, true, "getBlobContainerFromServicePrincipals",
                new Object[]{"testaccount", retryOptions},
                new Class<?>[]{String.class, RequestRetryOptions.class});
        } catch (Exception e) {
            // Expected - we're just testing that the method uses the default endpoint
            assertNotNull("Exception should not be null", e);
        }
    }

    @Test
    public void testGenerateUserDelegationKeySignedSasWithMockTime() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        // Test with specific time values
        BlockBlobClient mockBlobClient = mock(BlockBlobClient.class);
        OffsetDateTime specificTime = OffsetDateTime.parse("2023-12-31T23:59:59Z");

        try {
            provider.generateUserDelegationKeySignedSas(
                    mockBlobClient,
                    mock(com.azure.storage.blob.sas.BlobServiceSasSignatureValues.class),
                    specificTime
            );
        } catch (Exception e) {
            // Expected for authentication failure, but we're testing the time handling
            assertNotNull("Exception should not be null", e);
        }
    }

    @Test
    public void testGetBlobContainerWithNullRetryOptions() throws DataStoreException {
        String connectionString = getConnectionString();

        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();

        // Test with null retry options and empty properties
        BlobContainerClient containerClient = provider.getBlobContainer(null, new Properties());
        assertNotNull("Container client should not be null", containerClient);
        assertEquals("Container name should match", CONTAINER_NAME, containerClient.getBlobContainerName());
    }

    @Test
    public void testGetBlobContainerWithEmptyProperties() throws DataStoreException {
        String connectionString = getConnectionString();

        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();

        // Test with empty properties
        Properties emptyProperties = new Properties();
        BlobContainerClient containerClient = provider.getBlobContainer(new RequestRetryOptions(), emptyProperties);
        assertNotNull("Container client should not be null", containerClient);
        assertEquals("Container name should match", CONTAINER_NAME, containerClient.getBlobContainerName());
    }

    @Test
    public void testBuilderWithNullValues() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(null)
                .withAccountName(null)
                .withAccountKey(null)
                .withBlobEndpoint(null)
                .withSasToken(null)
                .withTenantId(null)
                .withClientId(null)
                .withClientSecret(null)
                .build();

        assertNotNull("Provider should not be null", provider);
        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
        assertNull("Connection string should be null", provider.getAzureConnectionString());
    }

    @Test
    public void testInitializeWithPropertiesNullProperties() {
        // Test with null properties object
        try {
            AzureBlobContainerProvider.Builder.builder(CONTAINER_NAME)
                    .initializeWithProperties(null);
            fail("Should throw NullPointerException with null properties");
        } catch (NullPointerException e) {
            // Expected
            assertNotNull("Exception should not be null", e);
        }
    }

    @Test
    public void testAuthenticationPriorityOrder() throws Exception {
        // Test that connection string takes priority over other authentication methods
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(getConnectionString())
                .withAccountName("testaccount")
                .withAccountKey("testkey")
                .withSasToken("sas-token")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        // Should use connection string path
        BlobContainerClient containerClient = provider.getBlobContainer();
        assertNotNull("Container client should not be null", containerClient);
        assertEquals("Container name should match", CONTAINER_NAME, containerClient.getBlobContainerName());
    }

    @Test
    public void testGetBlobContainerWithAccountKeyFallback() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withAccountKey("testkey")
                .withBlobEndpoint("https://testaccount.blob.core.windows.net")
                .build();

        try {
            BlobContainerClient containerClient = provider.getBlobContainer();
            assertNotNull("Container client should not be null", containerClient);
        } catch (Exception e) {
            // Expected for invalid credentials
            assertTrue("Should be DataStoreException", e instanceof DataStoreException);
        }
    }

    @Test
    public void testAuthenticateViaServicePrincipalTrue() throws Exception {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        // Use reflection to test private method
        boolean result = (boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertTrue("Should authenticate via service principal", result);
    }

    @Test
    public void testAuthenticateViaServicePrincipalFalseWithConnectionString() throws Exception {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString("connection-string")
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        // Use reflection to test private method
        boolean result = (boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertFalse("Should not authenticate via service principal when connection string is present", result);
    }

    @Test
    public void testAuthenticateViaServicePrincipalFalseWithMissingCredentials() throws Exception {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                // Missing client secret
                .build();

        // Use reflection to test private method
        boolean result = (boolean) MethodUtils.invokeMethod(provider, true, "authenticateViaServicePrincipal");
        assertFalse("Should not authenticate via service principal with missing credentials", result);
    }

    @Test
    public void testGetClientSecretCredential() throws Exception {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        // Use reflection to test private method
        ClientSecretCredential credential = (ClientSecretCredential) MethodUtils.invokeMethod(provider, true, "getClientSecretCredential");
        assertNotNull("Credential should not be null", credential);
    }

    @Test
    public void testGetBlobContainerFromServicePrincipals() throws Exception {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        RequestRetryOptions retryOptions = new RequestRetryOptions();

        // Use reflection to test private method
        try {
            BlobContainerClient containerClient = (BlobContainerClient) MethodUtils.invokeMethod(provider, true,
                "getBlobContainerFromServicePrincipals",
                new Object[]{"testaccount", retryOptions},
                new Class<?>[]{String.class, RequestRetryOptions.class});
            assertNotNull("Container client should not be null", containerClient);
        } catch (Exception e) {
            // Expected for invalid credentials
            assertNotNull("Exception should not be null", e);
        }
    }

    @Test
    public void testGenerateUserDelegationKeySignedSas() {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withTenantId("tenant-id")
                .withClientId("client-id")
                .withClientSecret("client-secret")
                .build();

        // Mock dependencies
        BlockBlobClient mockBlobClient = mock(BlockBlobClient.class);
        OffsetDateTime expiryTime = OffsetDateTime.now().plusHours(1);

        try {
            String sas = provider.generateUserDelegationKeySignedSas(
                    mockBlobClient,
                    mock(com.azure.storage.blob.sas.BlobServiceSasSignatureValues.class),
                    expiryTime
            );
            // This will likely fail due to authentication, but we're testing the method structure
            assertNotNull("SAS should not be null", sas);
        } catch (Exception e) {
            // Expected for invalid credentials or mock limitations
            assertNotNull("Exception should not be null", e);
        }
    }

    @Test
    public void testGenerateSas() throws Exception {
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAccountName("testaccount")
                .withAccountKey("testkey")
                .build();

        // Mock dependencies
        BlockBlobClient mockBlobClient = mock(BlockBlobClient.class);
        when(mockBlobClient.generateSas(any(), any())).thenReturn("mock-sas-token");

        // Use reflection to test private method
        String sas = (String) MethodUtils.invokeMethod(provider, true, "generateSas",
                new Object[]{mockBlobClient, mock(com.azure.storage.blob.sas.BlobServiceSasSignatureValues.class)},
                new Class<?>[]{BlockBlobClient.class, com.azure.storage.blob.sas.BlobServiceSasSignatureValues.class});
        assertEquals("SAS should match mock", "mock-sas-token", sas);
    }

    @Test
    public void testBuilderFieldAccess() throws Exception {
        AzureBlobContainerProvider.Builder builder = AzureBlobContainerProvider.Builder.builder(CONTAINER_NAME);

        // Test all builder methods and verify fields are set correctly
        builder.withAzureConnectionString("conn-string")
               .withAccountName("account")
               .withBlobEndpoint("endpoint")
               .withSasToken("sas")
               .withAccountKey("key")
               .withTenantId("tenant")
               .withClientId("client")
               .withClientSecret("secret");

        AzureBlobContainerProvider provider = builder.build();

        // Use reflection to verify all fields are set
        assertEquals("Container name should match", CONTAINER_NAME, provider.getContainerName());
        assertEquals("Connection string should match", "conn-string", provider.getAzureConnectionString());

        // Test private fields using reflection
        Field accountNameField = AzureBlobContainerProvider.class.getDeclaredField("accountName");
        accountNameField.setAccessible(true);
        assertEquals("Account name should match", "account", accountNameField.get(provider));

        Field blobEndpointField = AzureBlobContainerProvider.class.getDeclaredField("blobEndpoint");
        blobEndpointField.setAccessible(true);
        assertEquals("Blob endpoint should match", "endpoint", blobEndpointField.get(provider));

        Field sasTokenField = AzureBlobContainerProvider.class.getDeclaredField("sasToken");
        sasTokenField.setAccessible(true);
        assertEquals("SAS token should match", "sas", sasTokenField.get(provider));

        Field accountKeyField = AzureBlobContainerProvider.class.getDeclaredField("accountKey");
        accountKeyField.setAccessible(true);
        assertEquals("Account key should match", "key", accountKeyField.get(provider));

        Field tenantIdField = AzureBlobContainerProvider.class.getDeclaredField("tenantId");
        tenantIdField.setAccessible(true);
        assertEquals("Tenant ID should match", "tenant", tenantIdField.get(provider));

        Field clientIdField = AzureBlobContainerProvider.class.getDeclaredField("clientId");
        clientIdField.setAccessible(true);
        assertEquals("Client ID should match", "client", clientIdField.get(provider));

        Field clientSecretField = AzureBlobContainerProvider.class.getDeclaredField("clientSecret");
        clientSecretField.setAccessible(true);
        assertEquals("Client secret should match", "secret", clientSecretField.get(provider));
    }

    private String getConnectionString() {
        return String.format("DefaultEndpointsProtocol=http;AccountName=%s;AccountKey=%s;BlobEndpoint=%s",
                AzuriteDockerRule.ACCOUNT_NAME,
                AzuriteDockerRule.ACCOUNT_KEY,
                azurite.getBlobEndpoint());
    }

    /**
     * Test that generateSharedAccessSignature works without headers (backward compatibility).
     */
    @Test
    public void testGenerateSharedAccessSignatureWithoutHeaders() throws Exception {
        String connectionString = getConnectionString();
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();

        // Create container and blob
        BlobContainerClient container = azurite.getContainer(CONTAINER_NAME, connectionString);
        String blobName = "test-blob";
        container.getBlobClient(blobName).upload(
                com.azure.core.util.BinaryData.fromString("test content"), true);

        // Generate SAS without headers
        BlobSasPermission permissions = new BlobSasPermission().setReadPermission(true);
        String sas = provider.generateSharedAccessSignature(
                null, blobName, permissions, 3600, new Properties());

        assertNotNull("SAS should not be null", sas);
        assertTrue("SAS should contain signature", sas.contains("sig="));
        assertTrue("SAS should contain expiry", sas.contains("se="));
    }

    /**
     * Test that generateSharedAccessSignature includes headers when provided.
     */
    @Test
    public void testGenerateSharedAccessSignatureWithHeaders() throws Exception {
        String connectionString = getConnectionString();
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();

        // Create container and blob
        BlobContainerClient container = azurite.getContainer(CONTAINER_NAME, connectionString);
        String blobName = "test-blob-with-headers";
        container.getBlobClient(blobName).upload(
                com.azure.core.util.BinaryData.fromString("test content"), true);

        // Generate SAS with headers
        BlobSasPermission permissions = new BlobSasPermission().setReadPermission(true);
        BlobSasHeaders headers = new BlobSasHeaders()
                .setCacheControl("private, max-age=3600, immutable")
                .setContentType("image/png")
                .setContentDisposition("attachment; filename=\"test.png\"");

        String sas = provider.generateSharedAccessSignature(
                null, blobName, permissions, 3600, new Properties(), headers);

        assertNotNull("SAS should not be null", sas);
        assertTrue("SAS should contain signature", sas.contains("sig="));
        assertTrue("SAS should contain expiry", sas.contains("se="));

        // Verify headers are encoded in SAS
        // Azure encodes headers as rscc (cache-control), rsct (content-type), rscd (content-disposition)
        assertTrue("SAS should contain cache-control parameter",
                sas.contains("rscc=") || sas.contains("&rscc"));
        assertTrue("SAS should contain content-type parameter",
                sas.contains("rsct=") || sas.contains("&rsct"));
        assertTrue("SAS should contain content-disposition parameter",
                sas.contains("rscd=") || sas.contains("&rscd"));
    }

    /**
     * Test that generateSharedAccessSignature handles null headers gracefully.
     */
    @Test
    public void testGenerateSharedAccessSignatureWithNullHeaders() throws Exception {
        String connectionString = getConnectionString();
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();

        // Create container and blob
        BlobContainerClient container = azurite.getContainer(CONTAINER_NAME, connectionString);
        String blobName = "test-blob-null-headers";
        container.getBlobClient(blobName).upload(
                com.azure.core.util.BinaryData.fromString("test content"), true);

        // Generate SAS with null headers
        BlobSasPermission permissions = new BlobSasPermission().setReadPermission(true);
        String sas = provider.generateSharedAccessSignature(
                null, blobName, permissions, 3600, new Properties(), null);

        assertNotNull("SAS should not be null", sas);
        assertTrue("SAS should contain signature", sas.contains("sig="));
        assertTrue("SAS should contain expiry", sas.contains("se="));
    }

    /**
     * Test that generateSharedAccessSignature includes only provided headers.
     */
    @Test
    public void testGenerateSharedAccessSignatureWithPartialHeaders() throws Exception {
        String connectionString = getConnectionString();
        provider = AzureBlobContainerProvider.Builder
                .builder(CONTAINER_NAME)
                .withAzureConnectionString(connectionString)
                .build();

        // Create container and blob
        BlobContainerClient container = azurite.getContainer(CONTAINER_NAME, connectionString);
        String blobName = "test-blob-partial-headers";
        container.getBlobClient(blobName).upload(
                com.azure.core.util.BinaryData.fromString("test content"), true);

        // Generate SAS with only content-type header
        BlobSasPermission permissions = new BlobSasPermission().setReadPermission(true);
        BlobSasHeaders headers = new BlobSasHeaders().setContentType("application/json");

        String sas = provider.generateSharedAccessSignature(
                null, blobName, permissions, 3600, new Properties(), headers);

        assertNotNull("SAS should not be null", sas);
        assertTrue("SAS should contain signature", sas.contains("sig="));
        assertTrue("SAS should contain content-type parameter",
                sas.contains("rsct=") || sas.contains("&rsct"));
    }
}
